<?php

namespace App\Http\Controllers;

use App\Models\Commission;
use App\Models\Deal;
use App\Models\User;
use App\Models\Lender;
use App\Models\Setting;
use App\Models\UserTeam;
use App\Models\TeamLeader;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;

class CommissionController extends Controller
{
    public function index(Request $request)
    {

        $query = Commission::with(['deal.team.leader', 'user', 'lender', 'loan']);

        if (!Auth::user()->hasRole('Admin')) {
            $query->where('user_id', Auth::id());
        }

        if ($request->filled('lender_id')) {
            $query->where('lender_id', $request->lender_id);
        }
        $selectedUserId = $request->filled('user_id') ? (int) $request->user_id : null;
        if (!$selectedUserId && $request->filled('user_id')) {
        }
        if ($request->filled('user_status') && !$selectedUserId) {
            $query->where('status', $request->user_status);
        }
        // lender_payment_status filtering is applied later based on computed Remainder (loan total_commision - commission amount_paid).

        $dateFrom = $request->input('date_from');
        $dateTo = $request->input('date_to');
        if ($dateFrom || $dateTo) {
            $query->whereHas('loan', function ($q) use ($dateFrom, $dateTo) {
                if ($dateFrom && $dateTo) {
                    $q->whereBetween('date_funded', [\Carbon\Carbon::parse($dateFrom)->startOfDay(), \Carbon\Carbon::parse($dateTo)->endOfDay()]);
                } elseif ($dateFrom) {
                    $q->where('date_funded', '>=', \Carbon\Carbon::parse($dateFrom)->startOfDay());
                } elseif ($dateTo) {
                    $q->where('date_funded', '<=', \Carbon\Carbon::parse($dateTo)->endOfDay());
                }
            });
        }

        $paidFrom = $request->input('paid_from');
        $paidTo = $request->input('paid_to');
        if (!$selectedUserId && ($paidFrom || $paidTo)) {
            if ($paidFrom && $paidTo) {
                $query->whereBetween('paid_at', [\Carbon\Carbon::parse($paidFrom)->startOfDay(), \Carbon\Carbon::parse($paidTo)->endOfDay()]);
            } elseif ($paidFrom) {
                $query->where('paid_at', '>=', \Carbon\Carbon::parse($paidFrom)->startOfDay());
            } elseif ($paidTo) {
                $query->where('paid_at', '<=', \Carbon\Carbon::parse($paidTo)->endOfDay());
            }
        }

        $search = trim((string) $request->input('search', ''));
        if ($search !== '') {
            $query->where(function ($q) use ($search) {
                $q->whereHas('deal', function ($qq) use ($search) {
                    $qq->where('company_name', 'like', "%{$search}%");
                })
                ->orWhereHas('lender', function ($qq) use ($search) {
                    $qq->where('company_name', 'like', "%{$search}%")
                       ->orWhere('name', 'like', "%{$search}%");
                })
                ->orWhereHas('user', function ($qq) use ($search) {
                    $qq->where('name', 'like', "%{$search}%");
                })
                ->orWhereHas('loan', function ($qq) use ($search) {
                    $qq->where('loan_number', 'like', "%{$search}%");
                });
            });
        }

        $commissions = $query->orderBy('created_at', 'desc')->get();

        $leaderInfoByDeal = [];
        $byDeal = $commissions->groupBy('deal_id');
        foreach ($byDeal as $dealId => $items) {
            $deal = $items->first()->deal;
            $leaderUser = null;

            if ($deal && $deal->team) {
                $leaderUser = $deal->team->leader instanceof \Illuminate\Support\Collection
                    ? $deal->team->leader->first()
                    : $deal->team->leader()->first();
            }

            if (!$leaderUser) {
                $repUser = optional($items->first())->user;
                if ($repUser) {
                    $userTeam = UserTeam::where('user_id', $repUser->id)->first();
                    if ($userTeam) {
                        $teamLeaderPivot = TeamLeader::where('team_id', $userTeam->team_id)->with('leader')->first();
                        if ($teamLeaderPivot && $teamLeaderPivot->leader) {
                            $leaderUser = $teamLeaderPivot->leader;
                        }
                    }
                }
            }

            $leaderId = $leaderUser ? $leaderUser->id : null;
            $leaderName = $leaderUser ? ($leaderUser->name ?? trim(($leaderUser->first_name ?? '').' '.($leaderUser->last_name ?? ''))) : null;
            $leaderCm = null;
            if ($leaderId) {
                $leaderCommission = $items->firstWhere('user_id', $leaderId);
                if ($leaderCommission) {
                    $leaderCm = $leaderCommission->amount;
                }
            }
            $leaderInfoByDeal[$dealId] = [
                'leader_id' => $leaderId,
                'leader_name' => $leaderName,
                'leader_cm' => $leaderCm,
            ];
        }

        $lenders = Lender::all();
        $users = User::all();

        $displayCommissions = $commissions;
        if ($selectedUserId) {
            $pfStart = $paidFrom ? \Carbon\Carbon::parse($paidFrom)->startOfDay() : null;
            $pfEnd   = $paidTo ? \Carbon\Carbon::parse($paidTo)->endOfDay() : null;
            $userStatus = $request->input('user_status');

            $displayCommissions = $commissions->filter(function ($c) use ($selectedUserId, $leaderInfoByDeal, $userStatus, $pfStart, $pfEnd) {
                $isSr = ((int)$c->user_id) === (int)$selectedUserId;
                $leaderId = $leaderInfoByDeal[$c->deal_id]['leader_id'] ?? null;
                $isTl = $leaderId && ((int)$leaderId === (int)$selectedUserId);
                if (!$isSr && !$isTl) return false;

                $matchComponent = function ($flag, $paidAt) use ($userStatus, $pfStart, $pfEnd) {
                    if ($userStatus === 'paid' && !$flag) return false;
                    if ($userStatus === 'unpaid' && $flag) return false;
                    if (($pfStart || $pfEnd)) {
                        if (!$paidAt) return false;
                        $d = $paidAt instanceof \Carbon\Carbon ? $paidAt->copy() : \Carbon\Carbon::parse($paidAt);
                        if ($pfStart && $d->lt($pfStart)) return false;
                        if ($pfEnd && $d->gt($pfEnd)) return false;
                    }
                    return true;
                };

                $srOk = $isSr ? $matchComponent((bool)($c->sales_rep_paid ?? false), $c->sr_paid_at ?? null) : false;
                $tlOk = $isTl ? $matchComponent((bool)($c->team_leader_paid ?? false), $c->tl_paid_at ?? null) : false;

                return $srOk || $tlOk;
            });
        }

        // Apply lender_payment_status filter based on Remainder (loan total_commision - commission amount_paid)
        if ($request->filled('lender_payment_status')) {
            $lps = $request->lender_payment_status;
            $displayCommissions = $displayCommissions->filter(function ($c) use ($lps) {
                $loan = $c->loan;
                $totalCm = null;
                if ($loan && isset($loan->total_commision) && is_numeric($loan->total_commision)) {
                    $totalCm = (float) $loan->total_commision;
                }
                if ($totalCm === null) {
                    return false;
                }
                $amountPaid = is_numeric($c->amount_paid ?? null) ? (float) $c->amount_paid : 0.0;
                $remainder = $totalCm - $amountPaid;
                if ($lps === 'paid') {
                    return abs($remainder) < 0.01; // treat near-zero as paid
                } elseif ($lps === 'unpaid') {
                    return $amountPaid <= 0;
                } elseif ($lps === 'partial') {
                    return $amountPaid > 0 && $remainder > 0.01;
                }
                return true;
            });
        }

        $uniqueLoans = $displayCommissions->pluck('loan')->filter()->unique('id');
        $totalCommission = $uniqueLoans->sum(function ($loan) {
            return is_numeric($loan->total_commision ?? null) ? (float) $loan->total_commision : 0.0;
        });

        if ($selectedUserId) {
            $pfStart = $paidFrom ? \Carbon\Carbon::parse($paidFrom)->startOfDay() : null;
            $pfEnd   = $paidTo ? \Carbon\Carbon::parse($paidTo)->endOfDay() : null;
            $userStatus = $request->input('user_status');
            $componentMatches = function ($flag, $paidAt) use ($userStatus, $pfStart, $pfEnd) {
                if ($userStatus === 'paid' && !$flag) return false;
                if ($userStatus === 'unpaid' && $flag) return false;
                if (($pfStart || $pfEnd)) {
                    if (!$paidAt) return false;
                    $d = $paidAt instanceof \Carbon\Carbon ? $paidAt->copy() : \Carbon\Carbon::parse($paidAt);
                    if ($pfStart && $d->lt($pfStart)) return false;
                    if ($pfEnd && $d->gt($pfEnd)) return false;
                }
                return true;
            };
            $totalCommissionUser = $displayCommissions->sum(function ($c) use ($selectedUserId, $leaderInfoByDeal, $componentMatches) {
                $sum = 0.0;
                $isSr = ((int)$c->user_id) === (int)$selectedUserId;
                $leaderId = $leaderInfoByDeal[$c->deal_id]['leader_id'] ?? null;
                $isTl = $leaderId && ((int)$leaderId === (int)$selectedUserId);
                if ($isSr && $componentMatches((bool)($c->sales_rep_paid ?? false), $c->sr_paid_at ?? null)) {
                    $sum += is_numeric($c->sales_rep_cm ?? null) ? (float) $c->sales_rep_cm : 0.0;
                }
                if ($isTl && $componentMatches((bool)($c->team_leader_paid ?? false), $c->tl_paid_at ?? null)) {
                    $sum += is_numeric($c->team_leader_cm ?? null) ? (float) $c->team_leader_cm : 0.0;
                }
                return $sum;
            });
        } else {
            $totalCommissionUser = $displayCommissions->sum(function ($c) {
                return is_numeric($c->amount_paid ?? null) ? (float) $c->amount_paid : 0.0;
            });
        }

        $totalCommissionLender = $totalCommission; // Using loan total commission as lender-side amount
        $countDeals = $displayCommissions->unique('deal_id')->count();

        $commissions = $displayCommissions;

        return view('commission.index', compact(
            'commissions', 'lenders', 'users', 'leaderInfoByDeal',
            'totalCommission', 'totalCommissionLender', 'totalCommissionUser', 'countDeals'
        ));
    }

    public function toggleCmPaid(Request $request)
    {
        $data = $request->validate([
            'id'    => ['required','integer','exists:commissions,id'],
            'field' => ['required','in:sales_rep_paid,team_leader_paid'],
            'value' => ['required','boolean'],
        ]);
        $commission = Commission::findOrFail($data['id']);
        $commission->{$data['field']} = (bool) $data['value'];
        if ($data['field'] === 'sales_rep_paid') {
            $commission->sr_paid_at = $data['value'] ? now() : null;
        } elseif ($data['field'] === 'team_leader_paid') {
            $commission->tl_paid_at = $data['value'] ? now() : null;
        }
        $commission->save();
        return response()->json([
            'ok' => true,
            'id' => $commission->id,
            'field' => $data['field'],
            'value' => (bool) $commission->{$data['field']},
        ]);
    }

    public function markPaid(Request $request)
    {
        $commission = Commission::findOrFail($request->id);
        $commission->status = 'paid';
        $commission->amount_paid = $request->amount_paid;
        $commission->paid_at = now();
        $commission->save();
        return back()->with('success', 'Commission marked as paid.');
    }

    public function markPartial(Request $request)
    {
        $commission = Commission::findOrFail($request->id);

        $addAmount = is_numeric($request->amount_paid) ? (float) $request->amount_paid : 0.0;
        if ($addAmount < 0) {
            $addAmount = 0.0; // prevent negative additions
        }
        $existingPaid = is_numeric($commission->amount_paid) ? (float) $commission->amount_paid : 0.0;
        $commission->amount_paid = $existingPaid + $addAmount;

        $commission->status = 'partial';
        $commission->paid_at = now();
        $commission->save();
        return back()->with('success', 'Commission marked as partial paid.');
    }
}
