<?php
namespace App\Http\Controllers;

use Illuminate\Http\Request;
use App\Models\User;
use App\Models\TeamLeader;
use App\Models\UserTeam;
use App\Models\Team;

class UserController extends Controller
{
    public function index()
    {
        $users = User::all();
        return view('users.index', compact('users'));
    }

    // Show user data for editing (AJAX)

//    public function edit($id)
//    {
//        $user = User::findOrFail($id);
//        $roles = $user->roles->pluck('name');
//        $permissions = $user->getAllPermissions()->pluck('name');
//        return response()->json([
//            'user' => $user,
//            'roles' => $roles,
//            'permissions' => $permissions
//        ]);
//    }

    // Update user (AJAX)
//    public function update(Request $request, $id)
//    {
//        $user = User::findOrFail($id);
//        $user->first_name = $request->first_name;
//        $user->last_name = $request->last_name;
//        $user->email = $request->email;
//        $user->email2 = $request->email2;
//        $user->phone = $request->phone;
//        $user->commission_rate = $request->commission_rate;
//        $user->save();
//        // Update roles
//        if ($request->has('roles')) {
//            $user->syncRoles($request->roles);
//        }
//        // Update permissions
//        if ($request->has('permissions')) {
//            $user->syncPermissions($request->permissions);
//        }
//        return response()->json(['success' => true]);
//    }

    // Delete user (AJAX)
    public function destroy($id)
    {
        $user = User::findOrFail($id);
        $user->delete();
        return response()->json(['success' => true]);
    }

    // Toggle user status (AJAX)
    public function toggleStatus($id)
    {
        $user = User::findOrFail($id);
        $user->user_status = $user->user_status === 'active' ? 'disabled' : 'active';
        $user->save();
        return response()->json(['success' => true, 'status' => $user->user_status]);
    }



    public function allRolesPermissions()
    {
        // Если используешь spatie/laravel-permission:
        $roles = \Spatie\Permission\Models\Role::select('name')->get();
        $permissions = \Spatie\Permission\Models\Permission::select('name')->get();

        // Team Leaders list: users who either have Team_Lead role or appear in team_leader table
        $leadersFromRole = User::role('Team_Lead')->pluck('id');
        $leadersFromPivot = TeamLeader::distinct()->pluck('leader_id');
        $leaderIds = $leadersFromRole->merge($leadersFromPivot)->unique()->values();

        $teamLeaders = User::whereIn('id', $leaderIds)
            ->select('id','first_name','last_name','name','email')
            ->orderBy('first_name')
            ->get()
            ->map(function($u){
                $display = trim(($u->first_name ?? '').' '.($u->last_name ?? '')) ?: ($u->name ?: $u->email);
                return [ 'id' => $u->id, 'name' => $display ];
            });

        return response()->json([
            'roles' => $roles,
            'permissions' => $permissions,
            'team_leaders' => $teamLeaders,
        ]);
    }

// GET /users/{id}/edit
    public function edit($id)
    {
        $user = User::findOrFail($id);


        // Try to infer a team leader for this user from their team memberships (if any)
        $teamIds = UserTeam::where('user_id', $user->id)->pluck('team_id');
        $leaderId = null;
        if ($teamIds && $teamIds->count() > 0) {
            $leaderId = TeamLeader::whereIn('team_id', $teamIds)->value('leader_id');
        }

        if (request()->ajax()) {
            return response()->json([
                'user'            => $user,
                'roles'           => $user->getRoleNames()->toArray(),
                'permissions'     => $user->getPermissionNames()->toArray(),
                'team_leader_id'  => $leaderId,
            ]);
        }

        return view('users.edit', compact('user')); // если где-то нужен Blade
    }

// POST /users
    public function store(Request $request)
    {

        $data = $request->validate([
            'first_name' => ['nullable','string','max:255'],
            'last_name' => ['nullable','string','max:255'],
            'email' => ['required','email','max:255','unique:users,email'],
            'email2' => ['nullable','email','max:255'],
            'phone' => ['nullable','string','max:50'],
            'commission_rate' => ['nullable','numeric'],
            'team_leader_commission_rate' => ['nullable','numeric'],
            'password' => ['required','string','min:8','confirmed'],
            'roles' => ['array'],
            'roles.*' => ['string'],
            'permissions' => ['array'],
            'permissions.*' => ['string'],
            'team_leader_id' => ['nullable','integer','exists:users,id'],
        ]);



        $user = new User();
        $user->first_name = $data['first_name'] ?? null;
        $user->last_name  = $data['last_name']  ?? null;

        $user->name = trim(($data['first_name'] ?? '') . ' ' . ($data['last_name'] ?? ''));

        $user->email        = $data['email'];
        $user->email2       = $data['email2'] ?? null;
        $user->phone        = $data['phone'] ?? null;
        $user->commission_rate = $data['commission_rate'] ?? null;
        $user->team_leader_commission_rate = $data['team_leader_commission_rate'] ?? null;
        $user->password     = bcrypt($data['password']);
        $user->save();

        if (!empty($data['roles'])) {
            $user->syncRoles($data['roles']);
        }
        if (!empty($data['permissions'])) {
            $user->syncPermissions($data['permissions']);
        }

        if ($user->hasRole('Team_Lead')) {
            $this->ensureLeaderTeam($user);
        }

        if (!empty($data['team_leader_id'])) {
            $this->syncUserTeamByLeader($user->id, (int) $data['team_leader_id']);
        }

        return response()->json(['success' => true]);
    }

// PUT /users/{id}
    public function update(Request $request, $id)
    {
        $user = User::findOrFail($id);
        $data = $request->validate([
            'first_name' => ['nullable','string','max:255'],
            'last_name' => ['nullable','string','max:255'],
            'email' => ['required','email','max:255','unique:users,email,'.$user->id],
            'email2' => ['nullable','email','max:255'],
            'phone' => ['nullable','string','max:50'],
            'commission_rate' => ['nullable','numeric'],
            'team_leader_commission_rate' => ['nullable','numeric'],
            'roles' => ['array'],
            'roles.*' => ['string'],
            'permissions' => ['array'],
            'permissions.*' => ['string'],
            'password' => ['nullable','string','min:8','confirmed'],
            'team_leader_id' => ['nullable','integer','exists:users,id'],
        ]);

        $user->fill([
            'first_name' => $data['first_name'] ?? null,
            'last_name' => $data['last_name'] ?? null,
            'email' => $data['email'],
            'email2' => $data['email2'] ?? null,
            'phone' => $data['phone'] ?? null,
            'commission_rate' => $data['commission_rate'] ?? null,
            'team_leader_commission_rate' => $data['team_leader_commission_rate'] ?? null,
        ]);

        if (!empty($data['password'])) $user->password = bcrypt($data['password']);
        $user->name = trim(($data['first_name'] ?? '') . ' ' . ($data['last_name'] ?? ''));

        $user->save();

        if (isset($data['roles'])) {
            $user->syncRoles($data['roles']);
        }
        if (isset($data['permissions'])) {
            $user->syncPermissions($data['permissions']);
        }

        if ($user->hasRole('Team_Lead')) {
            $this->ensureLeaderTeam($user);
        }

        if (array_key_exists('team_leader_id', $data)) {

            if (!empty($data['team_leader_id'])) {
                $this->syncUserTeamByLeader($user->id, (int) $data['team_leader_id']);
            } else {
                $leaderTeamIds = TeamLeader::pluck('team_id');
                if ($leaderTeamIds && $leaderTeamIds->count() > 0) {
                    UserTeam::where('user_id', $user->id)->whereIn('team_id', $leaderTeamIds)->delete();
                }
            }
        }

        return response()->json(['success' => true]);
    }

    private function buildTeamName(User $user): string
    {
        $display = trim(($user->first_name ?? '') . ' ' . ($user->last_name ?? ''));
        if ($display === '') {
            $display = $user->name ?: $user->email;
        }
        return 'Team - ' . $display;
    }

    private function ensureLeaderTeam(User $user): void
    {

        $existingPivot = TeamLeader::where('leader_id', $user->id)->first();
        if ($existingPivot) {
            $team = Team::find($existingPivot->team_id);
            if (!$team) {
                $team = Team::create([
                    'name' => $this->buildTeamName($user),
                    'team_leader_id' => $user->id,
                ]);
                $existingPivot->team_id = $team->id;
                $existingPivot->save();
            }


            UserTeam::firstOrCreate([
                'team_id' => $existingPivot->team_id,
                'user_id' => $user->id,
            ]);
            return;
        }


        $team = Team::where('team_leader_id', $user->id)->first();
        if (!$team) {
            $team = Team::create([
                'name' => $this->buildTeamName($user),
                'team_leader_id' => $user->id
            ]);
        }

        TeamLeader::firstOrCreate([
            'team_id' => $team->id,
            'leader_id' => $user->id
        ]);


        UserTeam::firstOrCreate([
            'team_id' => $team->id,
            'user_id' => $user->id
        ]);
    }

    private function syncUserTeamByLeader(int $userId, ?int $leaderId): void
    {

        if (empty($leaderId)) return;

        $teamId = TeamLeader::where('leader_id', $leaderId)->orderBy('team_id')->value('team_id');

        if (!$teamId) return;

        $leaderTeamIds = TeamLeader::pluck('team_id')->unique()->toArray();
        if (!empty($leaderTeamIds)) {
            UserTeam::where('user_id', $userId)
                ->whereIn('team_id', $leaderTeamIds)
                ->where('team_id', '!=', $teamId)
                ->delete();
        }

        UserTeam::firstOrCreate([
            'team_id' => $teamId,
            'user_id' => $userId,
        ]);
    }
}