<?php

namespace App\Services;

use App\Models\Deal;
use App\Models\User;
use App\Models\Commission;
use App\Models\Setting;
use App\Models\Loan;
use App\Models\UserTeam;
use App\Models\TeamLeader;

class CommissionService
{
    /**
     * Calculate and create commission records for a funded deal.
     * Delegates to createForLoan for the most recent funded loan on the deal.
     */
    public function calculateForDeal(Deal $deal): void
    {
        $loan = Loan::where('deal_id', $deal->id)
            ->orderByDesc('date_funded')
            ->first();

        if ($loan) {
            $this->createForLoan($loan);
        }
    }


    public function createForLoan(Loan $loan): void
    {
        $deal = Deal::find($loan->deal_id);
        if (!$deal) {
            return;
        }

        $salesman = $deal->sale_rep ? User::where('name', $deal->sale_rep)->first() : null;
        if (!$salesman) {
            return;
        }

        $salesmanRate = is_numeric($salesman->commission_rate)
            ? (float) $salesman->commission_rate
            : 25.0;

        $total = (float) $deal->request_amount;
        $salesmanAmount = $total * $salesmanRate / 100.0;

        // Idempotent per Loan
        $already = Commission::where('loan_id', $loan->id)->exists();
        if ($already) {
            return;
        }

        $totalCm = is_numeric($loan->total_commision ?? null) ? (float) $loan->total_commision : 0.0;

        $salesRepRate = is_numeric($salesman->commission_rate) ? (float) $salesman->commission_rate : 0.0;
        $salesRepRate = $salesRepRate > 1 ? $salesRepRate / 100.0 : $salesRepRate;

        $leaderUser = null;
        if ($deal && $deal->team) {
            $leaderRel = $deal->team->leader;
            $leaderUser = $leaderRel instanceof \Illuminate\Support\Collection
                ? $leaderRel->first()
                : $deal->team->leader()->first();
        }
        if (!$leaderUser && $salesman) {
            $userTeam = UserTeam::where('user_id', $salesman->id)->first();
            if ($userTeam) {
                $teamLeaderPivot = TeamLeader::where('team_id', $userTeam->team_id)->with('leader')->first();
                if ($teamLeaderPivot && $teamLeaderPivot->leader) {
                    $leaderUser = $teamLeaderPivot->leader;
                }
            }
        }
        $leaderRate = $leaderUser && is_numeric($leaderUser->team_leader_commission_rate)
            ? (float) $leaderUser->team_leader_commission_rate
            : 0.0;
        $leaderRate = $leaderRate > 1 ? $leaderRate / 100.0 : $leaderRate;

        $salesRepCm   = (float) sprintf('%.2f', $totalCm * $salesRepRate);
        $teamLeaderCm = (float) sprintf('%.2f', $totalCm * $leaderRate);

        try {
            Commission::create([
                'deal_id'        => $deal->id,
                'user_id'        => $salesman->id,
                'lender_id'      => $loan->lender_id,
                'loan_id'        => $loan->id,
                'amount'         => $salesmanAmount,
                'sales_rep_cm'   => $salesRepCm,
                'team_leader_cm' => $teamLeaderCm,
                'status'         => 'unpaid',
            ]);
        } catch (\Illuminate\Database\QueryException $e) {
            // Ignore duplicate key on unique(loan_id)
            if (stripos($e->getMessage(), 'commissions_loan_id_unique') !== false) {
                return;
            }
            throw $e;
        }
    }
}
