#!/usr/bin/env bash
# Laravel deploy helper for cPanel/SSH
# Usage:
#   bash deploy.sh [--migrate] [--npm] [--no-down] [--no-optim]
# Flags:
#   --migrate  : run "php artisan migrate --force"
#   --npm      : run frontend build ("npm ci/install" + "npm run build/prod")
#   --no-down  : do NOT enable maintenance mode during deploy
#   --no-optim : skip config/route caches and "artisan optimize"

set -euo pipefail

### === Settings (adjust if needed) ===
APP_DIR="$(pwd)"                            # Laravel project root
PHP_BIN="${PHP_BIN:-php}"                   # e.g. /opt/cpanel/ea-php82/root/usr/bin/php
COMPOSER_BIN="${COMPOSER_BIN:-composer}"    # e.g. /opt/cpanel/composer/bin/composer
NPM_BIN="${NPM_BIN:-npm}"                   # npm path if not in PATH
ARTISAN="$PHP_BIN artisan"
LOG_DIR="$APP_DIR/storage/logs"
LOG_FILE="$LOG_DIR/deploy_$(date +%Y-%m-%d_%H-%M-%S).log"

RUN_MIGRATIONS=false
RUN_NPM=false
USE_DOWN=true
USE_OPTIM=true

### === Parse flags ===
for arg in "$@"; do
  case "$arg" in
    --migrate) RUN_MIGRATIONS=true ;;
    --npm) RUN_NPM=true ;;
    --no-down) USE_DOWN=false ;;
    --no-optim) USE_OPTIM=false ;;
    *) echo "Unknown flag: $arg"; exit 1 ;;
  esac
done

### === Prepare logging ===
mkdir -p "$LOG_DIR"
exec > >(tee -a "$LOG_FILE") 2>&1

echo "==> Deploy started: $(date)"
echo "Project: $APP_DIR"
echo "PHP: $PHP_BIN | Composer: $COMPOSER_BIN | NPM: $NPM_BIN"
echo "Log file: $LOG_FILE"

on_error() {
  echo "ERROR: Deploy failed. See log: $LOG_FILE"
  if $USE_DOWN; then
    echo "Attempting to exit maintenance mode..."
    $ARTISAN up || true
  fi
}
trap on_error ERR

cd "$APP_DIR"

### === Maintenance mode ===
if $USE_DOWN; then
  echo "==> Enabling maintenance mode"
  $ARTISAN down || true
else
  echo "==> Maintenance mode disabled by --no-down"
fi

### === Composer install (vendors) ===
if command -v "$COMPOSER_BIN" >/dev/null 2>&1; then
  echo "==> composer install --no-dev --optimize-autoloader"
  $COMPOSER_BIN install --no-dev --optimize-autoloader --no-interaction --prefer-dist
else
  echo "WARNING: Composer not found at '$COMPOSER_BIN'. Skipping vendor install."
fi

### === Migrations (optional) ===
if $RUN_MIGRATIONS; then
  echo "==> php artisan migrate --force"
  $ARTISAN migrate --force
else
  echo "==> Migrations skipped (no --migrate)"
fi

### === Clear caches ===
echo "==> Clearing caches (config, route, view, app cache)"
$ARTISAN cache:clear || true
$ARTISAN config:clear || true
$ARTISAN route:clear || true
$ARTISAN view:clear || true

### === Rebuild caches / optimize (optional) ===
if $USE_OPTIM; then
  echo "==> Rebuilding caches (config:cache, route:cache, optimize)"
  $ARTISAN config:cache || true
  $ARTISAN route:cache || true
  $ARTISAN optimize || true
else
  echo "==> Optimization skipped (--no-optim)"
fi

### === Frontend build (optional) ===
if $RUN_NPM; then
  if command -v "$NPM_BIN" >/dev/null 2>&1; then
    echo "==> Installing frontend dependencies"
    if [ -f package-lock.json ]; then
      $NPM_BIN ci
    else
      $NPM_BIN install
    fi

    # Detect build script without jq dependency
    if [ -f package.json ] && grep -q '"build"' package.json; then
      echo "==> npm run build"
      $NPM_BIN run build
    elif [ -f package.json ] && grep -q '"prod"' package.json; then
      echo "==> npm run prod"
      $NPM_BIN run prod
    else
      echo "WARNING: No \"build\" or \"prod\" script found in package.json. Skipping frontend build."
    fi
  else
    echo "WARNING: npm not found at '$NPM_BIN'. Skipping frontend build."
  fi
else
  echo "==> Frontend build skipped (no --npm)"
fi

### === Permissions ===
echo "==> Setting permissions for storage and bootstrap/cache"
chmod -R 755 storage bootstrap/cache || true

### === Exit maintenance mode ===
if $USE_DOWN; then
  echo "==> Disabling maintenance mode"
  $ARTISAN up || true
fi

echo "✅ Deploy finished successfully: $(date)"
