<?php

namespace App\Http\Controllers\Finance;

use App\Http\Controllers\Controller;
use App\Mail\LenderPosition;
use App\Models\Deal;
use App\Models\DealFiles;
use App\Models\DealLender;
use App\Models\UserTeam;
use Illuminate\Http\Request;
use App\Models\Lender;
use App\Models\SystemNote;
use App\Models\Team;
use App\Models\User;
use App\Models\Commission;
use App\Models\Loan;
use Carbon\Carbon;
use App\Models\EmailData;
use Illuminate\Support\Facades\Mail;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Arr;
use App\Mail\LenderEmail;


class LenderController extends Controller
{
    // Normalize a single string or array of emails into a unique list of valid addresses
    private function parseRecipientEmails($value): array
    {
        // Accepts a string with any mix of separators, display names, or angle brackets
        // and extracts a unique list of valid RFC email addresses.
        $haystack = '';
        if (is_array($value)) {
            $haystack = implode(' ', array_map(static function ($v) {
                return is_string($v) ? $v : (is_scalar($v) ? (string)$v : '');
            }, $value));
        } else {
            $haystack = (string) $value;
        }

        // Strip common prefixes like mailto:
        $haystack = str_replace(['mailto:', 'MAILTO:'], '', $haystack);

        // Find all email-like tokens anywhere in the string (handles Name <email@x>)
        preg_match_all('/[A-Z0-9._%+\-]+@[A-Z0-9.\-]+\.[A-Z]{2,}/i', $haystack, $m);
        $candidates = $m[0] ?? [];

        // Normalize, validate, unique
        $out = [];
        foreach ($candidates as $addr) {
            $addr = trim(strtolower($addr));
            if ($addr !== '' && filter_var($addr, FILTER_VALIDATE_EMAIL)) {
                $out[$addr] = true; // use keys to ensure uniqueness
            }
        }
        return array_keys($out);
    }

    public function index(Request $request)
    {
        Log::info('LenderController@index called', ['request' => $request->all()]);
        $q = Lender::query();
        if ($request->filled('company_name'))     $q->where('company_name', 'LIKE', '%'.$request->company_name.'%');
        if ($request->filled('primary_contact'))  $q->where('primary_contact', 'LIKE', '%'.$request->primary_contact.'%');
        if ($request->filled('phone'))            $q->where('phone', 'LIKE', '%'.$request->phone.'%');
        if ($request->filled('email'))            $q->where('email', 'LIKE', '%'.$request->email.'%');
        if ($request->filled('last_contacted'))   $q->whereDate('last_contacted', $request->last_contacted);

        $lenders = $q->get();

        $reps  = User::role('Sale_rep')->get();
        $teamL = User::role('Team_Lead')->get();
        $teams = Team::all();
        return view('lender.index', compact('lenders', 'reps', 'teamL','teams'));
    }



    public function editJson($id)
    {
        Log::info('LenderController@editJson called', ['id' => (string)$id]);
        $lender = Lender::findOrFail($id);

        return response()->json([
            'id'                      => $lender->id,
            'company_name'            => $lender->company_name,
            'primary_contact'         => $lender->primary_contact,
            'phone'                   => $lender->phone,
            'email'                   => $lender->email,
            'email2'                  => $lender->email2,
            'last_contacted'          => optional($lender->last_contacted)->toDateString(),
            'min_time_in_business'    => $lender->min_time_in_business,
            'min_revenue'             => $lender->min_revenue,
            'max_revenue'             => $lender->max_revenue,
            'accepts_defaults'        => (bool) $lender->accepts_defaults,
            'min_credit_score'        => $lender->min_credit_score,
            'max_nsf_per_month'       => $lender->max_nsf_per_month,
            'min_deposits_per_month'  => $lender->min_deposits_per_month,
            'excluded_industries'     => $lender->excluded_industries ?: [],
            'excluded_states'         => $lender->excluded_states ?: [],
            'has_integrated_api'      => (bool) $lender->has_integrated_api,
            'api_key'                 => $lender->api_key,
            'submission_type'         => $lender->submission_type === 'api' ? 'api' : 'email',
            'positions'               => $lender->positions ?: [],
            'status'                  => $lender->status,
        ]);
    }

    public function update(Request $request)
    {
        Log::info('LenderController@update called', ['request' => $request->all()]);
    
        $lender = Lender::findOrFail($request->id);
    
        $validated = $request->validate([
            'id'                      => 'required|exists:lenders,id',
            'company_name'            => 'required|string|max:255',
            'primary_contact'         => 'required|string|max:255',
            'phone'                   => 'required|string|max:50',
            'email'                   => 'required|email',
            'last_contacted'          => 'nullable|date',
    
            // Allow comma-separated emails
            'email2' => [
                'nullable',
                function ($attribute, $value, $fail) {
                    if ($value) {
                        $emails = array_map('trim', explode(',', $value));
                        foreach ($emails as $email) {
                            if (!filter_var($email, FILTER_VALIDATE_EMAIL)) {
                                return $fail("Each email in $attribute must be a valid email address.");
                            }
                        }
                    }
                }
            ],
    
            'positions'               => 'nullable|array',
            'positions.*'             => 'in:1,2,3,4,5,6,7,8,9',
            'min_time_in_business'    => 'nullable|integer|min:0',
            'min_revenue'             => 'nullable|integer|min:0',
            'max_revenue'             => 'nullable|integer|min:0',
            'accepts_defaults'        => 'nullable|boolean',
            'min_credit_score'        => 'nullable|integer|min:0|max:850',
            'max_nsf_per_month'       => 'nullable|integer|min:0',
            'min_deposits_per_month'  => 'nullable|integer|min:0',
            'excluded_industries'     => 'nullable|array',
            'excluded_industries.*'   => 'string',
            'excluded_states'         => 'nullable|array',
            'excluded_states.*'       => 'string|size:2',
            'has_integrated_api'      => 'nullable|boolean',
            'api_key'                 => 'nullable|string|max:255',
            'submission_type'         => 'nullable|in:email,api',
        ]);
    
        // Normalize booleans + arrays
        $validated['positions']           = array_values($request->input('positions', []));
        $validated['excluded_industries'] = array_values($request->input('excluded_industries', []));
        $validated['excluded_states']     = array_values($request->input('excluded_states', []));
        $validated['accepts_defaults']    = $request->boolean('accepts_defaults');
        $validated['has_integrated_api']  = $request->boolean('has_integrated_api');
    
        $lender->update($validated);
    
        return response()->json(['success' => true]);
    }


    public function show(Lender $lender)
    {
        return view('lender.show', compact('lender'));
    }

    // kept for compatibility
    public function edit(Request $request, Lender $lender)
    {
        return $lender;
    }

    public function destroy(Request $request)
    {
        $id = $request->id;
        $lender = Lender::find($id);
        if ($lender) {
            $user = User::where('email', $lender->email)->first();
            if ($user) $user->delete();
            $lender->delete();
            echo 'success';
        } else {
            echo 'fail';
        }
    }

    public function assign_lender(Request $request)
    {

        if (is_null($request->lenders)) {
            return redirect()->back()->with('error', 'Please Select Lender');
        }
        $lenders = $request->lenders;
        $deal = Deal::find($request->deal_id);
        $lender_deal = $deal->lenders->pluck('id')->toArray();

        foreach ($lenders as $lender_id) {
            $lender = Lender::find($lender_id);
            if (!in_array($lender->id, $lender_deal)) {
                $deal->lenders()->attach($lender_id, [
                    'old'       => false,
                    'date_sent' => date('Y-m-d'),
                    'status'    => 'Pending',
                ]);
                SystemNote::create([
                    'deal_id' => $deal->id,
                    'details' => 'Lender ('.$lender->company_name .') added to deal at : '. Carbon::now()->toDayDateTimeString() . ' Current Status of Deal is  : Underwritter' ,
                    'user_id' => auth()->user()->id,
                ]);
            }
        }

//        Commission::create([
//            'deal_id' => $deal->id,
//            'user_id' => auth()->user()->id,
//            'lender_id' => $lender_id,
//            'amount' => $deal->request_amount,
//            'status' => 'unpaid'
//        ]);

        $deal->update(['status' => 'Underwritter']);
        return redirect()->back()->with('success', 'Lender Added To Deal');
    }

    public function assign_lender_new(Request $request)
    {
        $deal = Deal::find($request->deal_id);
        $lender_deal = $deal->lenders;



        foreach ($lender_deal as $lender) {
            if (!$lender->pivot->old) {
                $lender->pivot->update([
                    'old'      => 1,
                    'date_sent'=> date('Y-m-d'),
                ]);
            }
        }


        $deal->update(['status' => 'Underwritter']);
        $details = [
            'subject'          => "New Position in deal",
            'title'            => 'Deal need fund',
            'email'            => $deal->email,
            'phone'            => $deal->phone_number,
            'requested_amount' => $deal->request_amount,
            'url'              => url('/show_deal').'/'.$deal->id,
        ];

        $emdRaw = EmailData::where('type','proc_sub')->pluck('email')->toArray();
        $emd = array_values(array_filter($emdRaw, function ($addr) {
            $addr = trim((string)$addr);
            return $addr !== '' && filter_var($addr, FILTER_VALIDATE_EMAIL);
        }));
        if (!empty($emd)) {
            try { Mail::mailer('gmail')->to($emd)->send(new LenderPosition($details)); }
            catch (\Throwable $e) { \Log::warning('Mail send failed (assign_lender_new broadcast): '.$e->getMessage()); }
        }

        SystemNote::create([
            'deal_id' => $deal->id,
            'details' => 'New Position created in deal at : '. Carbon::now()->toDayDateTimeString() . ' Current Status of Deal is  : Underwritter' ,
            'user_id' => auth()->user()->id,
        ]);

        DB::transaction(function () use ($deal) {

            $existing = Loan::where('deal_id', $deal->id)
                ->lockForUpdate()
                ->get();

            if ($existing->isEmpty()) {
                return;
            }

            Loan::where('deal_id', $deal->id)
                ->where('status', '!=', 'Auto Closed')
                ->update(['status' => 'Auto Closed']);

            $now = now();
            $rows = $existing->map(function (Loan $loan) use ($now) {
                $data = Arr::except($loan->getAttributes(), ['id', 'created_at', 'updated_at']);
                $data['status']      = 'Pending';
                $data['created_at']  = $now;
                $data['updated_at']  = $now;
                return $data;
            })->all();

            Loan::insert($rows);
        });



        return redirect()->back()->with('success', 'New Position created');
    }

    public function store_notes(Request $request)
    {
        $pivot_lender = DealLender::find($request->lender_id);
        $pivot_lender->update(['note' => $request->note]);
        echo 'success';
    }

    public function store_amount(Request $request)
    {
        $pivot_lender = DealLender::find($request->lender_id);
        $pivot_lender->update(['amount' => $request->amount]);
        echo 'success';
    }

    public function change_status(Request $request)
    {
        $v = $request->validate([
            'lender_id'    => ['required','integer','exists:lenders,id'],
            'deal_id'      => ['required','integer','exists:deals,id'],
            'status'       => ['required','string','max:255'],

            'funded_date'  => ['required','date'],
            'lender_name'  => ['nullable','string','max:255'],
            'amount'       => ['required','numeric','min:0'],
            'term_length'  => ['required','integer','min:1'],
            'term_unit'    => ['required','in:days,weeks,months'],
            'points'       => ['required','numeric','min:0'],
            'rep'          => ['required','string','max:255'],
            'loan_type'    => ['required','string','max:255'],
            'requested_amount' => ['sometimes','nullable','numeric','min:0'],
            'balance'          => ['sometimes','nullable','numeric','min:0'],
        ]);

        $fundedAmount     = (float)$v['amount'];
        $pointsPct        = (float)$v['points'];
        $totalCommission  = round($fundedAmount * ($pointsPct / 100), 2);

//        $requestedAmount = $v['requested_amount'] ? (float)$v['requested_amount'] : null;
//        $balance = $v['balance']
//            ? (float)$v['balance']
//            : ($fundedAmount + $totalCommission);

        $requestedAmount = $request->filled('requested_amount')
            ? (float) $request->input('requested_amount')
            : null;

        $balance = $request->filled('balance')
            ? (float) $request->input('balance')
            : ($fundedAmount + $totalCommission);

        $dateFunded       = Carbon::parse($v['funded_date'])->toDateString();
        $dayOfWeek        = Carbon::parse($v['funded_date'])->format('l');
        $paymentsCount    = (int)$v['term_length'];
        $paymentFrequency = $v['term_unit'];

        $loanNumber = 'LN-' . now()->format('Ymd') . '-' . str_pad((string)$v['deal_id'], 6, '0', STR_PAD_LEFT);

        $deal = Deal::find($request->deal_id);
        $lender_deal = $deal->lenders;

        foreach ($lender_deal as $lender) {
            if (!$lender->pivot->old) {
                $lender->pivot->update([
                    'old'      => 1,
                    'date_sent'=> date('Y-m-d'),
                ]);
            }
        }

        $createdLoan = null;
        DB::transaction(function () use (
            &$createdLoan,
            $v, $dateFunded, $dayOfWeek,
            $fundedAmount, $totalCommission,
            $paymentsCount, $paymentFrequency, $loanNumber,
            $requestedAmount, $balance, $pointsPct
        ) {

            $lender = Lender::findOrFail($v['lender_id']);
            $deal   = Deal::findOrFail($v['deal_id']);

            $lender->update(['status' => $v['status']]);
            $deal->update(['status'   => $v['status']]);

            SystemNote::create([
                'deal_id' => $deal->id,
                'details' => 'Lender (' . ($lender->company_name ?? 'N/A') . ') status updated. Deal status -> '
                    . $v['status'] . ' at: ' . Carbon::now()->toDayDateTimeString(),
                'user_id' => auth()->id(),
            ]);

            $loan = Loan::create([
                'loan_number'        => $loanNumber,
                'deal_id'            => $deal->id,
                'lender_id'          => $lender->id,

                'funded_amount'      => $fundedAmount,
                'requested_amount'   => $requestedAmount,
                'balance'            => $balance,
                'rate'               => null,
                'total_amount'       => $fundedAmount + $totalCommission,

                'day_of_week'        => $dayOfWeek,
                'payment_frequency'  => $paymentFrequency,
                'total_commision'    => $totalCommission,
                'total_bonus'        => null,
                'bonus_office'       => null,
                'bonus_rep1'         => null,
                'commision_office'   => null,
                'payments'           => $paymentsCount,

                'points'           => $pointsPct,

                'date_funded'        => $dateFunded,
                'date_updated'       => Carbon::now()->toDateString(),

                'sale_rep'           => $v['rep'],
                'sale_rep2'          => null,

                'status'             => $v['status'],
                'note'               => $v['loan_type'],
                'history'            => null,
            ]);

            $createdLoan = $loan;
        });


        try {
            if ($createdLoan) {
                app(\App\Services\CommissionService::class)->createForLoan($createdLoan);
            }
        } catch (\Throwable $e) {
            \Log::warning('Commission creation failed: '.$e->getMessage());
        }

        return response()->json(['ok' => true]);
    }

    public function update_loan(Request $request)
    {
        $v = $request->validate([
            'loan_id'     => ['required','integer','exists:loans,id'],
            'lender_id'   => ['required','integer','exists:lenders,id'],
            'deal_id'     => ['required','integer','exists:deals,id'],
            'status'      => ['required','string','max:255'],

            'funded_date' => ['required','date'],
            'lender_name' => ['nullable','string','max:255'],
            'amount'      => ['required','numeric','min:0'],
            'term_length' => ['required','integer','min:1'],
            'term_unit'   => ['required','in:days,weeks,months'],
            'points'      => ['required','numeric','min:0'],
            'rep'         => ['required','string','max:255'],
            'loan_type'   => ['required','string','max:255'],
            'requested_amount' => ['sometimes','nullable','numeric','min:0'],
            'balance'          => ['sometimes','nullable','numeric','min:0'],
        ]);

        $loan = Loan::findOrFail($v['loan_id']);

        $fundedAmount    = (float)$v['amount'];
        $pointsPct       = (float)$v['points'];
        $totalCommission = round($fundedAmount * ($pointsPct / 100), 2);
        $dateFunded      = Carbon::parse($v['funded_date'])->toDateString();
        $dayOfWeek       = Carbon::parse($v['funded_date'])->format('l');
        $paymentsCount   = (int)$v['term_length'];
        $paymentFreq     = $v['term_unit'];

//        $requestedAmount = $v['requested_amount'] ? (float)$v['requested_amount'] : null;
//        $balance = $v['balance']
//            ? (float)$v['balance']
//            : ($fundedAmount + $totalCommission);

        $requestedAmount = $request->filled('requested_amount')
            ? (float) $request->input('requested_amount')
            : null;

        $balance = $request->filled('balance')
            ? (float) $request->input('balance')
            : ($fundedAmount + $totalCommission);

        DB::transaction(function () use (
            $loan, $v, $fundedAmount, $totalCommission,
            $dateFunded, $dayOfWeek, $paymentsCount,
            $paymentFreq, $requestedAmount, $balance, $pointsPct
        ) {

            $loan->update([
                'deal_id'           => $v['deal_id'],
                'lender_id'         => $v['lender_id'],

                'funded_amount'     => $fundedAmount,
                'total_commision'   => $totalCommission,
                'total_amount'      => $fundedAmount + $totalCommission,

                'requested_amount' => $requestedAmount,
                'balance'          => $balance,

                'points'           => $pointsPct,

                'day_of_week'       => $dayOfWeek,
                'payment_frequency' => $paymentFreq,
                'payments'          => $paymentsCount,

                'date_funded'       => $dateFunded,
                'date_updated'      => Carbon::now()->toDateString(),

                'sale_rep'          => $v['rep'],
                'status'            => $v['status'],
                'note'              => $v['loan_type'],
            ]);

            Lender::whereKey($v['lender_id'])->update(['status' => $v['status']]);
            Deal::whereKey($v['deal_id'])->update(['status' => $v['status']]);

            SystemNote::create([
                'deal_id' => $v['deal_id'],
                'details' => 'Loan #' . $loan->id . ' updated to status ' . $v['status'] . ' at ' . now()->toDayDateTimeString(),
                'user_id' => auth()->id(),
            ]);
        });

        return response()->json(['ok' => true]);
    }

    public function update_status(Request $request)
    {
        $lender = DealLender::find($request->lender_id);
        $lender->update(['status' => $request->status]);
        $deal = Deal::find($request->deal_id);

        if ($request->status == 'Approved') {
            $deal->update(['status' => 'Approved']);
        }

        $reshop = true;
        if ($request->status == 'Declined') {
            foreach ($deal->lenders as $l) {
                if ($l->pivot->status != 'Declined') $reshop = false;
            }
            if ($reshop) $deal->update(['status' => 'Reshop']);
        }
        if ($request->status != 'Approved' && $reshop == false) {
            $deal->update(['status' => 'Underwritter']);
            foreach ($deal->lenders as $l) {
                if ($l->pivot->status == 'Approved') $deal->update(['status' => 'Approved']);
            }
        }

        SystemNote::create([
            'deal_id' => $deal->id,
            'details' => 'Lender ( ' .$lender->company_name .' ) Status Updated , Deal Status updated to '. $deal->status .' at : '. Carbon::now()->toDayDateTimeString() ,
            'user_id' => auth()->user()->id,
        ]);

        $deals = Deal::all();
        $submitted    = $deals->where('status', 'Submitted');
        $underwritter = $deals->where('status', 'Underwritter');
        $approved     = $deals->where('status', 'Approved');
        $coldoffer    = $deals->where('status', 'Cold Offer');
        $reshopSet    = $deals->where('status', 'Reshop');
        $open         = $deals->where('status', 'Open');
        $team_id      = UserTeam::where('user_id',auth()->user()->id)->pluck('team_id')->toArray();
        $open_team    = $deals->where('status', 'Open For Team')->whereIn('team_id',$team_id);
        $dead         = $deals->where('status', 'Dead');
        $funded       = $deals->where('status', 'Funded');
        $doc          = $deals->where('status', "Doc's Out");
        $proc         = $deals->where('status', "Processor");

        return view('partials.deal_ajax',compact('submitted','underwritter','approved','coldoffer','reshopSet','open','open_team','dead','funded','doc','proc','deal'));
    }

    public function send_email(Request $request)
    {
        set_time_limit(0);



        // Find lender
        $lender = Lender::find($request->lender_id);
        if (!$lender) {
            Log::warning('send_email: lender not found', ['lender_id' => $request->lender_id]);
            return response()->json(['message' => 'Lender not found'], 404);
        }

        // Resolve deal reliably
        $deal = null;
        if ($request->filled('deal_id')) {
            $deal = Deal::find($request->deal_id);
        }
        // Try to parse from Referer like /show_deal/{id}
        if (!$deal) {
            $ref = $request->headers->get('referer');
            if ($ref && preg_match('#/show_deal/(\d+)#', $ref, $m)) {
                $deal = Deal::find((int)$m[1]);
            }
        }
        // Fallback: use the latest pivot that links this lender to a deal
        if (!$deal) {
            $pivot = DealLender::where('lender_id', $lender->id)->orderByDesc('id')->first();
            if ($pivot) {
                $deal = Deal::find($pivot->deal_id);
            }
        }

        if (!$deal) {
            Log::warning('send_email: deal not resolved', [
                'lender_id' => $lender->id,
                'request_deal_id' => $request->deal_id ?? null,
                'referer' => $request->headers->get('referer')
            ]);
            return response()->json(['message' => 'Deal not found for this lender'], 422);
        }

        $details = [
            'subject'          => 'New Deal - ' . ($deal->company_name ?? 'Unknown Company'),
            'title'            => 'Deal need fund',
            'email'            => $deal->email,
            'phone'            => $deal->phone_number,
            'requested_amount' => $deal->request_amount,
            'url'              => url('/show_deal').'/'.$deal->id,
        ];
        $mail = new LenderPosition($details);

        $files = DealFiles::where('deal_id', $deal->id)->get();
        if ($files->count() > 0) {
            foreach ($files as $file) {
                $pdfPath = public_path('files/'.$file->files);
                if (is_file($pdfPath)) {
                    $mail->attach($pdfPath);
                }
            }
        }

        $recipients = $this->parseRecipientEmails($lender->email2);

        if (!empty($recipients)) {
            try {
                Mail::mailer('gmail')->to($recipients)->send($mail);
                SystemNote::create([
                    'deal_id' => $deal->id,
                    'details' => 'Email to Lender (' . ($lender->company_name ?? 'Unknown Lender') . ') at ' . Carbon::now()->toDayDateTimeString() . ' | Recipient(s): ' . implode(', ', $recipients) . ' | Status: SUCCESS',
                    'user_id' => auth()->user()->id,
                ]);
                return response()->json([
                    'ok' => true,
                    'message' => 'Email sent successfully',
                    'recipients' => $recipients,
                    'mailer' => 'gmail',
                ]);
            } catch (\Throwable $e) {
                \Log::warning('Mail send failed (send_email to lender): '.$e->getMessage(), [
                    'exception' => get_class($e),
                    'file' => $e->getFile(),
                    'line' => $e->getLine(),
                    'code' => $e->getCode(),
                    'recipients' => $recipients,
                    'deal_id' => $deal->id,
                    'lender_id' => $lender->id,
                ]);
                SystemNote::create([
                    'deal_id' => $deal->id,
                    'details' => 'Email to Lender (' . ($lender->company_name ?? 'Unknown Lender') . ') at ' . Carbon::now()->toDayDateTimeString() . ' | Recipient(s): ' . implode(', ', $recipients) . ' | Status: FAILED | Error: ' . $e->getMessage(),
                    'user_id' => auth()->user()->id,
                ]);
                return response()->json([
                    'ok' => false,
                    'message' => 'Email send failed',
                    'error' => $e->getMessage(),
                ], 500);
            }
        } else {
            SystemNote::create([
                'deal_id' => $deal->id,
                'details' => 'Email to Lender (' . ($lender->company_name ?? 'Unknown Lender') . ') not sent | Status: FAILED | Reason: Invalid or missing Submission Email',
                'user_id' => auth()->user()->id,
            ]);
            return response()->json([
                'ok' => false,
                'message' => 'Invalid or missing recipient email(s) for lender',
            ], 422);
        }
        return response()->json(true);
    }

    public function send_email_all(Request $request)
    {
        $deal = Deal::find($request->deal_id);
        if ($deal->status=='Submitted') {
            $deal->update(['status' => 'Processor']);
        }
        $details = [
            'subject'          => $deal->company_name,
            'title'            => 'Deal need fund',
            'email'            => $deal->email,
            'phone'            => $deal->phone_number,
            'requested_amount' => $deal->request_amount,
            'url'              => url('/show_deal').'/'.$deal->id,
            'message'          => "Have a look at deals attachments below.",
        ];

        $mail = new LenderPosition($details);
        $files = DealFiles::where('deal_id', $deal->id)->get();
        if ($files->count() > 0) {
            foreach ($files as $file) {
                $pdfPath = public_path('files/'.$file->files);
                $mail->attach($pdfPath);
            }
        }

        $emdRaw = EmailData::where('type','proc_sub')->pluck('email')->toArray();
        $emd = array_values(array_filter($emdRaw, function ($addr) {
            $addr = trim((string)$addr);
            return $addr !== '' && filter_var($addr, FILTER_VALIDATE_EMAIL);
        }));
        if (!empty($emd)) {
            try { Mail::mailer('gmail')->to($emd)->send($mail); }
            catch (\Throwable $e) { \Log::warning('Mail send failed (send_email_all broadcast): '.$e->getMessage(), ['exception' => get_class($e)]); }
        }

        SystemNote::create([
            'deal_id' => $deal->id,
            'details' => 'Email Send to all Lender at : '. Carbon::now()->toDayDateTimeString() ,
            'user_id' => auth()->user()->id,
        ]);

        $deals = Deal::all();
        $submitted   = $deals->where('status', 'Submitted');
        $underwritter= $deals->where('status', 'Underwritter');
        $approved    = $deals->where('status', 'Approved');
        $coldoffer   = $deals->where('status', 'Cold Offer');
        $reshop      = $deals->where('status', 'Reshop');
        $open        = $deals->where('status', 'Open');
        $team_id     = UserTeam::where('user_id',auth()->user()->id)->pluck('team_id')->toArray();
        $open_team   = $deals->where('status', 'Open For Team')->whereIn('team_id',$team_id);
        $dead        = $deals->where('status', 'Dead');
        $funded      = $deals->where('status', 'Funded');
        $doc         = $deals->where('status', "Doc's Out");
        $proc        = $deals->where('status', "Processor");

        return view('partials.deal_ajax',compact('submitted','underwritter','approved','coldoffer','reshop','open','open_team','dead','funded','doc','proc','deal'));
    }

    public function send_email_all_selected(Request $request)
    {
        $lenderIds = $request->input('lender_ids', []);
        $dealId    = $request->input('deal_id');
    
        $deal = Deal::find($dealId);
    
        $results = [];
    
        foreach ($lenderIds as $lenderId) {
    
            $lender = Lender::find($lenderId);
    
            if (!$lender) {
                $results[$lenderId] = 'Lender not found';
                continue;
            }
    
            // Submission emails (email2)
            $rawEmails = trim($lender->email2);
    
            if (!$rawEmails) {
                $results[$lenderId] = "Missing submission email";
                continue;
            }
    
            // Support multiple comma-separated emails
            $emails = array_filter(array_map('trim', explode(',', $rawEmails)));
    
            if (count($emails) === 0) {
                $results[$lenderId] = "No valid emails found";
                continue;
            }
    
            try {
    
                // REQUIRED details for lender.blade.php
                $details = [
                    'subject'          => 'New Deal - ' . ($deal->company_name ?? 'Unknown Company'),
                    'company_name'     => $deal->company_name ?? '',     // <-- REQUIRED FIX
                    'title'            => 'Deal need fund',
                    'email'            => $deal->email,
                    'phone'            => $deal->phone_number,
                    'requested_amount' => $deal->request_amount,
                    'url'              => url('/show_deal/'.$deal->id),
                ];
    
                // Build mail
                $mail = new \App\Mail\LenderEmail($details);
    
                // Attach files
                $files = \App\Models\DealFiles::where('deal_id', $deal->id)->get();
                foreach ($files as $file) {
                    $pdfPath = public_path('files/'.$file->files);
                    if (is_file($pdfPath)) {
                        $mail->attach($pdfPath);
                    }
                }
    
                \Mail::mailer('gmail')->to($emails)->send($mail);
    
                $results[$lenderId] = "SENT to " . implode(", ", $emails);
    
            } catch (\Throwable $e) {
    
                \Log::error("Email send FAILED", [
                    'lender_id' => $lenderId,
                    'emails'    => $emails,
                    'error'     => $e->getMessage()
                ]);
    
                $results[$lenderId] = "FAILED: " . $e->getMessage();
            }
        }
    
        return response()->json([
            'ok' => true,
            'results' => $results
        ]);
    }

}