<?php

namespace App\Mail\Transport;

use GuzzleHttp\Client as HttpClient;
use Psr\Log\LoggerInterface;
use Swift_Events_EventListener;
use Swift_Mime_SimpleMessage;
use Swift_Transport;

class GmailApiTransport implements Swift_Transport
{
    /** @var bool */
    protected $started = false;

    /** @var HttpClient */
    protected $http;

    /** @var string|null */
    protected $accessToken;

    /** @var LoggerInterface|null */
    protected $logger;

    public function __construct(HttpClient $http, ?string $accessToken = null, ?LoggerInterface $logger = null)
    {
        $this->http = $http;
        $this->accessToken = $accessToken;
        $this->logger = $logger;
    }

    public function isStarted()
    {
        return $this->started;
    }

    public function start()
    {
        $this->started = true;
    }

    public function stop()
    {
        $this->started = false;
    }

    public function ping()
    {
        // Gmail API doesn't maintain a persistent connection; consider it always alive
        return true;
    }

    public function send(Swift_Mime_SimpleMessage $message, &$failedRecipients = null)
    {
        $failedRecipients = (array) $failedRecipients;

        // Ensure a From header is present (Gmail requires a valid sender)
        $this->ensureFromHeader($message);

        // Count all recipients (to, cc, bcc)
        $recipientCount = $this->getRecipientCount($message);

        $rawMessage = $message->toString();
        $raw = rtrim(strtr(base64_encode($rawMessage), '+/', '-_'), '=');

        // Get token or try to refresh if missing
        $token = $this->accessToken ?: config('services.gmail.access_token');
        if (!$token) {
            $token = $this->refreshAccessToken();
        }

        if (!$token) {
            if ($this->logger) {
                $this->logger->error('GmailApiTransport: Missing access token and refresh failed.');
            }
            throw new \Swift_TransportException('Gmail API access token is missing');
        }

        try {
            $response = $this->http->post('gmail/v1/users/me/messages/send', [
                'headers' => [
                    'Authorization' => 'Bearer ' . $token,
                    'Accept' => 'application/json',
                    'Content-Type' => 'application/json',
                ],
                'json' => [
                    'raw' => $raw,
                ],
                'http_errors' => false,
                'timeout' => 30,
            ]);

            $status = $response->getStatusCode();
            if ($status >= 200 && $status < 300) {
                return $recipientCount;
            }

            // If unauthorized, attempt a single refresh and retry
            if ($status === 401) {
                if ($this->logger) {
                    $this->logger->warning('GmailApiTransport: 401 Unauthorized, attempting token refresh...');
                }
                $newToken = $this->refreshAccessToken();
                if ($newToken) {
                    $response = $this->http->post('gmail/v1/users/me/messages/send', [
                        'headers' => [
                            'Authorization' => 'Bearer ' . $newToken,
                            'Accept' => 'application/json',
                            'Content-Type' => 'application/json',
                        ],
                        'json' => [
                            'raw' => $raw,
                        ],
                        'http_errors' => false,
                        'timeout' => 30,
                    ]);

                    $status = $response->getStatusCode();
                    if ($status >= 200 && $status < 300) {
                        return $recipientCount;
                    }
                }
            }

            $body = (string) $response->getBody();
            if ($this->logger) {
                $this->logger->error('GmailApiTransport: Gmail API send failed', [
                    'status' => $status,
                    'body' => $body,
                ]);
            }

            throw new \Swift_TransportException(sprintf('Gmail API send failed with status %d: %s', $status, $body));
        } catch (\Throwable $e) {
            if ($this->logger) {
                $this->logger->error('GmailApiTransport: Exception while sending', [
                    'exception' => $e->getMessage(),
                ]);
            }
            if (!($e instanceof \Swift_TransportException)) {
                throw new \Swift_TransportException($e->getMessage(), (int) ($e->getCode() ?: 0), $e);
            }
            throw $e;
        }
    }

    public function registerPlugin(Swift_Events_EventListener $plugin)
    {
        // No plugins supported for this custom transport at the moment
    }

    protected function getRecipientCount(Swift_Mime_SimpleMessage $message): int
    {
        $to = (array) $message->getTo();
        $cc = (array) $message->getCc();
        $bcc = (array) $message->getBcc();
        return count($to) + count($cc) + count($bcc);
    }

    private function ensureFromHeader(Swift_Mime_SimpleMessage $message): void
    {
        if (!$message->getFrom()) {
            $fromAddr = config('mail.from.address', 'no-reply@shullink.com');
            $fromName = config('mail.from.name', null);
            if ($fromAddr) {
                $message->setFrom([$fromAddr => $fromName ?: $fromAddr]);
            }
        }
    }

    private function getGmailConfigMerged(): array
    {
        $cfg = config('services.gmail') ?: [];
        $json = $cfg['oauth_json'] ?? null;
        if (is_string($json) && $json !== '') {
            try {
                $data = json_decode($json, true) ?: [];
                if (is_array($data)) {
                    $cfg = array_merge($cfg, $data);
                }
            } catch (\Throwable $e) {
                // ignore invalid json
            }
        }
        return $cfg;
    }

    private function refreshAccessToken(): ?string
    {
        $cfg = $this->getGmailConfigMerged();
        $refreshToken = $cfg['refresh_token'] ?? null;
        $clientId     = $cfg['client_id'] ?? null;
        $clientSecret = $cfg['client_secret'] ?? null;
        $tokenUri     = $cfg['token_uri'] ?? 'https://oauth2.googleapis.com/token';

        if (!$refreshToken || !$clientId || !$clientSecret) {
            if ($this->logger) {
                $this->logger->error('GmailApiTransport: Cannot refresh token (missing refresh_token, client_id, or client_secret).');
            }
            return null;
        }

        try {
            $resp = $this->http->post($tokenUri, [
                'headers' => [
                    'Accept' => 'application/json',
                ],
                'form_params' => [
                    'grant_type' => 'refresh_token',
                    'client_id' => $clientId,
                    'client_secret' => $clientSecret,
                    'refresh_token' => $refreshToken,
                ],
                'http_errors' => false,
                'timeout' => 30,
            ]);

            $status = $resp->getStatusCode();
            $body   = (string) $resp->getBody();
            $json   = json_decode($body, true);

            if ($status >= 200 && $status < 300 && is_array($json) && !empty($json['access_token'])) {
                $new = (string) $json['access_token'];
                $this->accessToken = $new;
                // Update runtime config for subsequent sends in this request
                try { config(['services.gmail.access_token' => $new]); } catch (\Throwable $e) { /* ignore */ }
                if ($this->logger) {
                    $this->logger->info('GmailApiTransport: Access token refreshed successfully.');
                }
                return $new;
            }

            if ($this->logger) {
                $this->logger->error('GmailApiTransport: Token refresh failed', [
                    'status' => $status,
                    'body' => $body,
                ]);
            }
        } catch (\Throwable $e) {
            if ($this->logger) {
                $this->logger->error('GmailApiTransport: Exception during token refresh', [
                    'exception' => $e->getMessage(),
                ]);
            }
        }

        return null;
    }
}
