<?php

namespace App\Http\Controllers;

use App\Models\Contact;
use App\Models\CustomUrl;
use App\Models\FieldMapping;
use App\Models\User;
use Exception;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Schema;
use Throwable;

use function PHPUnit\Framework\isNull;

class CustomWebhookController extends Controller
{
    public function __construct()
    {
        // Exclude the webhook handler from admin middleware so webhooks can post without auth
        $this->middleware(['role_or_permission:admin|view admin'])->except('handle');
    }

    public function createCustomUrl()
    {
        $urlId = random_int(111111, 999999);
        $columns = Schema::getColumnListing('contacts');
        $user_id = auth()->id();
        // Project context: try to get from session, fallback to null
        $project_id = session('pid', null); // 'pid' should be set when user is in a project context

        return view('admin.custom-url')
            ->with('id', $urlId)
            ->with('columns', $columns)
            ->with('user_id', $user_id)
            ->with('project_id', $project_id);
    }

    public function index()
    {
        $customUrls = CustomUrl::all();

        return view('admin.custom-url-table')->with('customUrls', $customUrls);
    }

    public function storeCustomUrl(Request $request)
    {
        // Validate the form data
        $validatedData = $request->validate([
            'name' => 'required',
            'fields.*' => 'required',
            'webfields.*' => 'required',
            'defaultvalues.*' => 'nullable',
        ]);

    // Create a new Custom Url model and fill it with the form data
    $customUrl = new CustomUrl();
    $customUrl->name = $validatedData['name'];
    $customUrl->description = $request->input('desc');
    $customUrl->posting_url = $request->input('url');
    $customUrl->url_id = $request->input('url_id');
    $customUrl->email = $request->input('email');
    $customUrl->update_existing_contacts = $request->has('updateContact');
    $customUrl->update_field = $request->updatefield;
    $customUrl->user_id = $request->input('user_id');
    $customUrl->project_id = $request->input('project_id');

    // Save the Custom Url model to the database
    $customUrl->save();

        // Loop through each field mapping and create a new Field Mapping model for each one
        for ($i = 0; $i < count($validatedData['fields']); $i++) {
            $fieldMapping = new FieldMapping();
            $fieldMapping->custom_url_id = $customUrl->id;
            $fieldMapping->salescallz_field = $validatedData['fields'][$i];
            $fieldMapping->web_form_field = $validatedData['webfields'][$i];
            $fieldMapping->default_value = $validatedData['defaultvalues'][$i];
            $fieldMapping->save();
        }
        $notification = array(
            'alert-type' => 'success',
            'message' => 'Custom url(webhook) Created Successfully'
        );
        return redirect()->back()->with($notification);
    }

    public function handle(Request $request)
    {



        // Find the CustomUrl model by url_id
        $customUrl = CustomUrl::where('url_id', $request->id)->first();

        if (!$customUrl)
            return response()->json(['message' => 'url not found']);
        // Get the FieldMapping models for the CustomUrl
        $fieldMappings = FieldMapping::where('custom_url_id', $customUrl->id)->get();

        // try {
        //code...

        // Get the data from the webhook request
        $webhookData = $request->all();

        // Loop through the FieldMappings and map the data to the Contact model
        $contactData = [];
        foreach ($fieldMappings as $fieldMapping) {
            $webFormField = $fieldMapping->web_form_field;
            if (isset($webhookData[$webFormField])) {
                $salescallzField = $fieldMapping->salescallz_field;
                $contactData[$salescallzField] = $webhookData[$webFormField];
            }
        }
    // Use the user_id and project_id from the CustomUrl
    $contactData['user_id'] = $customUrl->user_id;
    $contactData['project_id'] = $customUrl->project_id;

        // Check if update contact is enabled
        if ($customUrl->update_existing_contacts && $customUrl->update_field != null) {

            $email = $contactData[$customUrl->update_field] ?? null;
            if ($email) {
                // Find the existing contact by email   
                $existingContact = Contact::where($customUrl->update_field, $email)->first();
                if ($existingContact) {
                    // Update the existing contact with the new values
                    $existingContact->update($contactData);
                } else {
                    // Check if contact already exists with this email
                    $duplicateContact = Contact::where('email', $email)->first();
                    if ($duplicateContact) {
                        // Handle duplicate email address gracefully
                        return response()->json(['message' => 'Contact with this email already exists.']);
                    } else {
                        // Create a new contact with the provided values
                        Contact::create($contactData);
                    }
                }
            }
        } else {
            // Check if contact already exists with this email
            $email = $contactData['email'] ?? null;
            if ($email) {
                $duplicateContact = Contact::where('email', $email)->first();
                if ($duplicateContact) {
                    // Handle duplicate email address gracefully
                    return response()->json(['message' => 'Contact with this email already exists.']);
                }
            }
            // Create a new contact with the provided values
            Contact::create($contactData);
        }

        return response()->json(['message' => 'Webhook handled successfully.']);

        // } catch (\Throwable $th) {
        //     return response()->json(['message' => $th]);
        // }

    }

    public function edit($id)
    {

        $customUrl = CustomUrl::find($id);
        $columns = Schema::getColumnListing('contacts');

        return view('admin.edit-custom-url')->with('customUrl', $customUrl)->with('columns', $columns);

    }

    public function update(Request $request, $id)
    {
        // Validate the input fields
        $validatedData = $request->validate([
            'name' => 'required',
            'fields.*' => 'required',
            'webfields.*' => 'required',
            'defaultvalues.*' => 'nullable',
        ]);

        // Find the custom URL record in the database
        $customUrl = CustomUrl::findOrFail($id);

        // Update the custom URL record with the new form data
        $customUrl->name = $request->input('name');
        $customUrl->description = $request->input('desc');
        $customUrl->email = $request->input('email');
        $customUrl->update_existing_contacts = $request->has('updateContact');
        $customUrl->update_field = $request->input('updatefield');
        $customUrl->fieldMappings()->delete();

        foreach ($request->input('fields') as $key => $field) {
            $customUrl->fieldMappings()->create([
                'salescallz_field' => $field,
                'web_form_field' => $request->input('webfields')[$key],
                'default_value' => $request->input('defaultvalues')[$key],
            ]);
        }

        $customUrl->save();
        $notification = array(
            'alert-type' => 'success',
            'message' => 'Custom URL updated successfully.'
        );
        return redirect()->back()->with($notification);
    }

    public function destroy($id)
    {
        try {
            $contacts = CustomUrl::find($id);
            $contacts->delete();

            return redirect()->back()->with(['alert-type' => "success", 'message' => "Custom url Successfully delete"]);
        } catch (Exception $e) {
            return redirect()->back()->with(['alert-type' => "error", 'message' => "Something went wrong"]);
        } catch (Throwable $e) {
            return redirect()->back()->with(['alert-type' => "error", 'message' => "Something went wrong"]);
        }
    }
}